<?php
/**
 * Plugin Name: PostGap
 * Description: Helps identify gaps in your posting schedule with calendar and list views of published content. Shows empty days where no posts were published, allows quick draft creation to fill those gaps, and displays both published posts and drafts chronologically to maintain consistent posting frequency.
 * Version: 1.1.1
 * Author: Jorge Pereira
 * Author URI:   https://jorgep.com/blog/wordpress-plugins/
 */

if (!defined('ABSPATH')) exit;

function postgap_add_menu_item() {
    add_menu_page(
        'PostGap',
        'PostGap',
        'edit_posts',
        'postgap',
        'postgap_render_calendar_page',
        'dashicons-calendar-alt'
    );
}
add_action('admin_menu', 'postgap_add_menu_item');

function postgap_enqueue_styles() {
    wp_enqueue_style('postgap-styles', plugins_url('style.css', __FILE__));
}
add_action('admin_enqueue_scripts', 'postgap_enqueue_styles');

function postgap_create_draft_post() {
    check_ajax_referer('postgap_nonce', 'nonce');
    
    $date = sanitize_text_field($_POST['date']);
    $date_obj = new DateTime($date);
    
    $post_data = array(
        'post_status' => 'draft',
        'post_type' => 'post',
        'post_title' => $date_obj->format('F j, Y') . ' Post',
        'post_date' => $date . ' ' . current_time('H:i:s'),
        'post_date_gmt' => get_gmt_from_date($date . ' ' . current_time('H:i:s'))
    );
    
    $post_id = wp_insert_post($post_data);
    
    if ($post_id) {
        wp_send_json_success(array(
            'edit_url' => get_edit_post_link($post_id, '')
        ));
    } else {
        wp_send_json_error();
    }
}
add_action('wp_ajax_postgap_create_draft', 'postgap_create_draft_post');

function postgap_get_posts_for_date($date, $status) {
    return get_posts(array(
        'post_type' => 'post',
        'post_status' => $status,
        'posts_per_page' => -1,
        'date_query' => array(
            array(
                'year' => date('Y', strtotime($date)),
                'month' => date('m', strtotime($date)),
                'day' => date('d', strtotime($date)),
            ),
        ),
    ));
}

function postgap_get_draft_list_url($date) {
    return add_query_arg(array(
        'post_type' => 'post',
        'post_status' => 'draft',
        'year' => date('Y', strtotime($date)),
        'monthnum' => date('m', strtotime($date)),
        'day' => date('d', strtotime($date))
    ), admin_url('edit.php'));
}

function postgap_render_calendar_page() {
    if (!current_user_can('edit_posts')) {
        wp_die(__('You do not have sufficient permissions to access this page.'));
    }
    
    $month = isset($_GET['month']) ? intval($_GET['month']) : intval(date('m'));
    $year = isset($_GET['year']) ? intval($_GET['year']) : intval(date('Y'));
    $view = isset($_GET['view']) ? sanitize_text_field($_GET['view']) : 'calendar';
    
    $post_counts = array();
    $draft_counts = array();
    $posts_by_date = array();
    
    // Get all days in month

    $days_in_month = date('t', mktime(0, 0, 0, $month, 1, $year));
    for ($day = 1; $day <= $days_in_month; $day++) {
        $date = sprintf('%04d-%02d-%02d', $year, $month, $day);
        
        // Get published posts
        $posts = postgap_get_posts_for_date($date, 'publish');
        $post_counts[$day] = count($posts);
        
        // Get drafts
        $drafts = postgap_get_posts_for_date($date, 'draft');
        $draft_counts[$day] = count($drafts);
        
        // Store posts and drafts for list view
        $posts_by_date[$day] = array(
            'published' => $posts,
            'drafts' => $drafts
        );
    }
    

    
    $prev_month = $month - 1;
    $prev_year = $year;
    if ($prev_month < 1) {
        $prev_month = 12;
        $prev_year--;
    }
    
    $next_month = $month + 1;
    $next_year = $year;
    if ($next_month > 12) {
        $next_month = 1;
        $next_year++;
    }
    ?>
    <div class="wrap">
        <h1>PostGap</h1>
        
        <div class="calendar-nav">
            <a href="?page=postgap&month=<?php echo $prev_month; ?>&year=<?php echo $prev_year; ?>&view=<?php echo $view; ?>" class="button">&lt; Previous Month</a>
            <h2><?php echo date('F Y', mktime(0, 0, 0, $month, 1, $year)); ?></h2>
            <a href="?page=postgap&month=<?php echo $next_month; ?>&year=<?php echo $next_year; ?>&view=<?php echo $view; ?>" class="button">Next Month &gt;</a>
        </div>

        <div class="view-selector">
            <select id="view-select" onchange="window.location.href=this.value">
                <option value="?page=postgap&month=<?php echo $month; ?>&year=<?php echo $year; ?>&view=calendar" <?php selected($view, 'calendar'); ?>>Calendar View</option>
                <option value="?page=postgap&month=<?php echo $month; ?>&year=<?php echo $year; ?>&view=list" <?php selected($view, 'list'); ?>>List View</option>
            </select>
        </div>

        <?php if ($view === 'list'): ?>
            <div class="list-view">
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Posts</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php for ($day = 1; $day <= $days_in_month; $day++): 
                            $date = sprintf('%04d-%02d-%02d', $year, $month, $day);
                            $day_posts = $posts_by_date[$day];
                        ?>
                            <tr>
                                <td><?php echo date('F j, Y', strtotime($date)); ?></td>
                                <td>
                                    <?php if (!empty($day_posts['published'])): ?>
                                        <?php foreach ($day_posts['published'] as $post): ?>
                                            <div class="post-item published">
                                                <a href="<?php echo get_edit_post_link($post->ID); ?>"><?php echo esc_html($post->post_title); ?></a>
                                            </div>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                    
                                    <?php if (!empty($day_posts['drafts'])): ?>
                                        <?php foreach ($day_posts['drafts'] as $draft): ?>
                                            <div class="post-item draft">
                                                <a href="<?php echo get_edit_post_link($draft->ID); ?>"><?php echo esc_html($draft->post_title); ?> (Draft)</a>
                                            </div>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                    
                                    <?php if (empty($day_posts['published']) && empty($day_posts['drafts'])): ?>
                                        <button class="create-draft-btn button button-small" data-date="<?php echo $date; ?>">Create Draft</button>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endfor; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <table class="wp-list-table widefat fixed striped calendar-table">
                <thead>
                    <tr>
                        <?php
                        $days = array('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat');
                        foreach ($days as $day) {
                            echo "<th>$day</th>";
                        }
                        ?>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    $first_day = date('w', mktime(0, 0, 0, $month, 1, $year));
                    
                    $day_count = 1;
                    $cell_count = 0;
                    
                    while ($day_count <= $days_in_month) {
                        if ($cell_count % 7 == 0) {
                            echo "<tr>";
                        }
                        
                        if ($cell_count < $first_day) {
                            echo "<td class='empty'></td>";
                        } else {
                            $date = sprintf('%04d-%02d-%02d', $year, $month, $day_count);
                            $post_count = $post_counts[$day_count];
                            $draft_count = $draft_counts[$day_count];
                            
                            echo "<td>";
                            echo "<div class='day-number'>$day_count</div>";

                            if ($post_count > 0) {
                                echo "<div class='post-count'>$post_count published</div>";
                            } else {
                                echo "<button class='create-draft-btn button button-small' data-date='$date'>Create Draft</button>";
                            }

                            if ($draft_count > 0) {
                                $draft_list_url = postgap_get_draft_list_url($date);
                                echo "<div class='draft-count'><a href='$draft_list_url'>$draft_count draft" . ($draft_count > 1 ? "s" : "") . "</a></div>";
                            }

                            echo "</td>";
                            
                            $day_count++;
                        }
                        
                        $cell_count++;
                        
                        if ($cell_count % 7 == 0) {
                            echo "</tr>";
                        }
                    }
                    
                    while ($cell_count % 7 != 0) {
                        echo "<td class='empty'></td>";
                        $cell_count++;
                    }
                    ?>
                </tbody>
            </table>
        <?php endif; ?>
        
        <?php wp_nonce_field('postgap_nonce', 'postgap_nonce'); ?>
    </div>
    
    <script>
    jQuery(document).ready(function($) {
        $('.create-draft-btn').click(function() {
            var button = $(this);
            var date = button.data('date');
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'postgap_create_draft',
                    date: date,
                    nonce: $('#postgap_nonce').val()
                },
                success: function(response) {
                    if (response.success) {
                        window.location.href = response.data.edit_url;
                    }
                }
            });
        });
    });
    </script>
<?php
}
